<?php

/**
 * Plugin MP3 Player pour Bludit
 *
 * Paramètre "dir" accepte trois formats :
 *   - Chemin relatif Bludit  : "bl-content/uploads/mp3/"
 *   - Chemin absolu serveur  : "/Integrale/2009 - KaleïdoskPOP/"
 *   - URL complète           : "https://exemple.com/musique/"
 *
 * Shortcode dans une page :
 *   {mp3player}
 *   {mp3player dir="/Integrale/2009 - MonAlbum/" title="Mon Album"}
 *   {mp3player dir="/Integrale/2009 - MonAlbum/" title="Mon Album" autoplay="true"}
 */
class pluginMp3player extends Plugin {

    public function init()
    {
        $this->dbFields = array(
            'mp3_directory' => '',
            'player_title'  => 'Lecteur MP3',
            'autoplay'      => false,
        );
    }

    public function form()
    {
        $html  = '<div class="form-group">';
        $html .= '<label>Répertoire des fichiers MP3</label>';
        $html .= '<input type="text" name="mp3_directory" value="' . $this->getValue('mp3_directory') . '" class="form-control" />';
        $html .= '<small class="text-muted">';
        $html .= 'Trois formats acceptés :<br>';
        $html .= '• Chemin absolu serveur : <code>/Integrale/2009 - MonAlbum/</code><br>';
        $html .= '• Chemin relatif Bludit : <code>bl-content/uploads/mp3/</code><br>';
        $html .= '• URL complète : <code>https://exemple.com/musique/</code><br><br>';
        $html .= 'Shortcode dans une page : <code>{mp3player dir="/Integrale/MonAlbum/" title="Mon Titre"}</code><br>';
        $html .= 'Laisser vide pour ne rien afficher en sidebar.';
        $html .= '</small>';
        $html .= '</div>';

        $html .= '<div class="form-group">';
        $html .= '<label>Titre du lecteur</label>';
        $html .= '<input type="text" name="player_title" value="' . $this->getValue('player_title') . '" class="form-control" />';
        $html .= '</div>';

        $html .= '<div class="form-group">';
        $html .= '<label>';
        $html .= '<input type="checkbox" name="autoplay" value="true" ' . ($this->getValue('autoplay') ? 'checked' : '') . ' /> ';
        $html .= 'Lecture automatique';
        $html .= '</label>';
        $html .= '</div>';

        return $html;
    }

    public function post()
    {
        $this->dbFields['mp3_directory'] = $_POST['mp3_directory'];
        $this->dbFields['player_title']  = $_POST['player_title'];
        $this->dbFields['autoplay']      = isset($_POST['autoplay']) ? true : false;
        $this->save();
    }

    /**
     * Hook beforeSiteLoad : remplace {mp3player ...} dans le contenu de toutes
     * les pages chargées (page unique ET page d'accueil avec $content tableau).
     */
    public function beforeSiteLoad()
    {
        global $content, $page, $WHERE_AM_I;

        $self = $this;

        // Pattern shortcode : accepte {mp3player ...} même si Bludit a encodé les accolades
        $pattern = '/(?:\{|&#123;)mp3player([^}&#]*)(?:\}|&#125;)/iu';

        $callback = function($matches) use ($self) {
            $attrs    = array();
            $attrStr  = isset($matches[1]) ? $matches[1] : '';

            // Décoder les entités HTML que Bludit peut injecter (&quot; &#8220; &#8221; etc.)
            $attrStr = html_entity_decode($attrStr, ENT_QUOTES | ENT_HTML5, 'UTF-8');

            if (!empty($attrStr)) {
                // Accepte guillemets droits " ' ET typographiques \u201c\u201d\u2018\u2019
                preg_match_all(
                    '/(\w+)\s*=\s*(?:["\'\x{201C}\x{2018}])(.*?)(?:["\'\x{201D}\x{2019}])/iu',
                    $attrStr,
                    $m,
                    PREG_SET_ORDER
                );
                foreach ($m as $pair) {
                    $attrs[$pair[1]] = $pair[2];
                }
            }

            $dir      = isset($attrs['dir'])      ? trim($attrs['dir'])              : $self->getValue('mp3_directory');
            $title    = isset($attrs['title'])    ? trim($attrs['title'])             : $self->getValue('player_title');
            $autoplay = isset($attrs['autoplay']) ? ($attrs['autoplay'] === 'true')   : (bool)$self->getValue('autoplay');

            return $self->renderPlayer($dir, $title, $autoplay);
        };

        // Page d'accueil : $content est un tableau de pages
        if ($WHERE_AM_I === 'home' && !empty($content)) {
            foreach ($content as $p) {
                $raw = $p->content();
                if (strpos($raw, 'mp3player') !== false) {
                    $p->setField('content', preg_replace_callback($pattern, $callback, $raw));
                }
            }
            return;
        }

        // Page unique
        if (!empty($page) && method_exists($page, 'content')) {
            $raw = $page->content();
            if (strpos($raw, 'mp3player') !== false) {
                $page->setField('content', preg_replace_callback($pattern, $callback, $raw));
            }
        }
    }
    /**
     * Hook siteSidebar : affiche le lecteur en sidebar.
     * Si le répertoire est vide ou non configuré, n'affiche rien.
     */
    public function siteSidebar()
    {
        $dir = trim($this->getValue('mp3_directory'));

        // Rien configuré → silence total
        if ($dir === '') {
            return;
        }

        $result = $this->scanDir($dir);

        // Pas de fichiers ou erreur → silence total en sidebar (pas de message d'erreur visible)
        if (isset($result['error']) || empty($result['files'])) {
            return;
        }

        echo $this->buildPlayer($result['files'], $this->getValue('player_title'), (bool)$this->getValue('autoplay'), $dir);
    }

    /**
     * Génère le HTML du lecteur (appelé par shortcode ou sidebar).
     * En contexte shortcode, affiche un message d'erreur si problème.
     */
    public function renderPlayer($dir, $title, $autoplay)
    {
        $dir    = trim($dir);
        $result = $this->scanDir($dir);

        if (isset($result['error'])) {
            return '<p class="mp3player-empty">⚠ ' . htmlspecialchars($result['error']) . '</p>';
        }

        if (empty($result['files'])) {
            return '<p class="mp3player-empty">⚠ Aucun fichier MP3 trouvé dans : <code>' . htmlspecialchars($dir) . '</code></p>';
        }

        return $this->buildPlayer($result['files'], $title, $autoplay, $dir);
    }

    /**
     * Aiguillage : détecte le type de chemin et appelle le bon scanner.
     *
     * @param  string $dir  Chemin relatif Bludit, chemin absolu /... ou URL https://...
     * @return array        ['files' => [...]] ou ['error' => '...']
     */
    private function scanDir($dir)
    {
        $dir = trim($dir);

        if ($dir === '') {
            return array('error' => 'Aucun répertoire configuré.');
        }

        // URL complète http(s)://
        if (preg_match('#^https?://#i', $dir)) {
            return $this->scanRemoteDir($dir);
        }

        // Chemin absolu depuis la racine du serveur web : commence par /
        if (strpos($dir, '/') === 0) {
            return $this->scanAbsoluteDir($dir);
        }

        // Chemin relatif depuis la racine Bludit (PATH_ROOT)
        return $this->scanRelativeDir($dir);
    }

    /**
     * Chemin absolu depuis la racine du serveur (/Integrale/album/).
     * Construit le chemin fichier via $_SERVER['DOCUMENT_ROOT'].
     */
    private function scanAbsoluteDir($absPath)
    {
        $absPath = rtrim($absPath, '/') . '/';
        $docRoot = rtrim($_SERVER['DOCUMENT_ROOT'], '/');
        $fullPath = $docRoot . $absPath;

        if (!is_dir($fullPath)) {
            return array('error' => 'Répertoire introuvable : ' . $absPath);
        }

        $files = array();
        foreach (scandir($fullPath) as $item) {
            if (preg_match('/\.mp3$/i', $item)) {
                // URL publique : encode chaque segment du chemin
                $segments = explode('/', trim($absPath, '/'));
                $encodedPath = '/' . implode('/', array_map('rawurlencode', $segments)) . '/';
                $files[] = array(
                    'url'   => DOMAIN . $encodedPath . rawurlencode($item),
                    'label' => pathinfo($item, PATHINFO_FILENAME),
                );
            }
        }

        usort($files, function($a, $b) { return strcmp($a['label'], $b['label']); });

        return array('files' => $files);
    }

    /**
     * Chemin relatif depuis PATH_ROOT de Bludit (bl-content/uploads/mp3/).
     */
    private function scanRelativeDir($relDir)
    {
        $relDir  = rtrim($relDir, '/') . '/';
        $fullPath = PATH_ROOT . $relDir;

        if (!is_dir($fullPath)) {
            return array('error' => 'Répertoire introuvable : ' . $relDir);
        }

        $files = array();
        foreach (scandir($fullPath) as $item) {
            if (preg_match('/\.mp3$/i', $item)) {
                $files[] = array(
                    'url'   => DOMAIN . $relDir . rawurlencode($item),
                    'label' => pathinfo($item, PATHINFO_FILENAME),
                );
            }
        }

        usort($files, function($a, $b) { return strcmp($a['label'], $b['label']); });

        return array('files' => $files);
    }

    /**
     * URL distante : parse le listing HTML Apache/Nginx.
     */
    private function scanRemoteDir($url)
    {
        $url  = rtrim($url, '/') . '/';
        $html = $this->httpGet($url);

        if ($html === false) {
            return array('error' => 'Impossible d\'accéder à : ' . $url);
        }

        preg_match_all('/href=["\']([^"\'?#]+\.mp3)["\']/i', $html, $matches);

        if (empty($matches[1])) {
            return array('files' => array());
        }

        $files = array();
        $seen  = array();
        $parsed = parse_url($url);

        foreach ($matches[1] as $href) {
            if (preg_match('#^https?://#i', $href)) {
                $fileUrl = $href;
            } elseif (strpos($href, '/') === 0) {
                $fileUrl = $parsed['scheme'] . '://' . $parsed['host'] . $href;
            } else {
                $fileUrl = $url . $href;
            }

            $key = strtolower($fileUrl);
            if (isset($seen[$key])) continue;
            $seen[$key] = true;

            $basename = basename(rawurldecode($fileUrl));
            $files[]  = array(
                'url'   => $fileUrl,
                'label' => pathinfo($basename, PATHINFO_FILENAME),
            );
        }

        usort($files, function($a, $b) { return strcmp($a['label'], $b['label']); });

        return array('files' => $files);
    }

    /**
     * Requête HTTP GET (cURL puis file_get_contents en fallback).
     */
    private function httpGet($url)
    {
        if (function_exists('curl_init')) {
            $ch = curl_init();
            curl_setopt_array($ch, array(
                CURLOPT_URL            => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_TIMEOUT        => 10,
                CURLOPT_ENCODING       => '',
                CURLOPT_USERAGENT      => 'Mozilla/5.0 (compatible; BluditMp3Player/1.2)',
                CURLOPT_SSL_VERIFYPEER => true,
            ));
            $body = curl_exec($ch);
            $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            return ($body !== false && $code === 200) ? $body : false;
        }

        $ctx = stream_context_create(array(
            'http' => array('timeout' => 10, 'user_agent' => 'Mozilla/5.0'),
            'ssl'  => array('verify_peer' => true),
        ));
        return @file_get_contents($url, false, $ctx);
    }

    /**
     * Construit le HTML + JS du lecteur.
     */
    private function buildPlayer($files, $title, $autoplay, $dirKey)
    {
        $uid = 'mp3p_' . substr(md5($dirKey . $title), 0, 8);

        // SVG icons
        $svgPrev = '<svg viewBox="0 0 24 24"><polygon points="19,5 9,12 19,19"/><rect x="5" y="5" width="3" height="14"/></svg>';
        $svgPlay = '<svg viewBox="0 0 24 24"><polygon points="6,4 20,12 6,20"/></svg>';
        $svgPause= '<svg viewBox="0 0 24 24"><rect x="5" y="4" width="5" height="16"/><rect x="14" y="4" width="5" height="16"/></svg>';
        $svgNext = '<svg viewBox="0 0 24 24"><polygon points="5,5 15,12 5,19"/><rect x="16" y="5" width="3" height="14"/></svg>';

        ob_start();
        $this->printCSS();
        ?>
        <div class="mp3player" id="<?php echo $uid; ?>">

            <?php if ($title): ?>
            <div class="mp3player-title"><?php echo htmlspecialchars($title); ?></div>
            <?php endif; ?>

            <div class="mp3player-track" id="<?php echo $uid; ?>-track">—</div>

            <div class="mp3player-bar-wrap" id="<?php echo $uid; ?>-barwrap">
                <div class="mp3player-bar" id="<?php echo $uid; ?>-bar"></div>
            </div>

            <div class="mp3player-time">
                <span id="<?php echo $uid; ?>-cur">0:00</span>
                <span id="<?php echo $uid; ?>-dur">0:00</span>
            </div>

            <div class="mp3player-controls">
                <button class="mp3player-btn" id="<?php echo $uid; ?>-prev" title="Précédent">
                    <?php echo $svgPrev; ?>
                </button>
                <button class="mp3player-btn mp3player-btn-play" id="<?php echo $uid; ?>-play" title="Lecture / Pause">
                    <?php echo $svgPlay; ?>
                </button>
                <button class="mp3player-btn" id="<?php echo $uid; ?>-next" title="Suivant">
                    <?php echo $svgNext; ?>
                </button>
                <input class="mp3player-vol" type="range" id="<?php echo $uid; ?>-vol"
                       min="0" max="1" step="0.05" value="0.8" title="Volume" />
            </div>

            <ul class="mp3player-list" id="<?php echo $uid; ?>-list">
                <?php foreach ($files as $i => $f): ?>
                <li class="mp3player-item" data-src="<?php echo htmlspecialchars($f['url']); ?>">
                    <span class="mp3player-num"><?php echo $i + 1; ?>.</span>
                    <?php echo htmlspecialchars($f['label']); ?>
                </li>
                <?php endforeach; ?>
            </ul>

            <audio id="<?php echo $uid; ?>-audio"></audio>
        </div>

        <script>
        (function(){
            var uid=<?php echo json_encode($uid); ?>,ap=<?php echo $autoplay?'true':'false'; ?>;
            var svgPlay=<?php echo json_encode($svgPlay); ?>;
            var svgPause=<?php echo json_encode($svgPause); ?>;
            var audio=document.getElementById(uid+'-audio'),
                items=document.querySelectorAll('#'+uid+'-list .mp3player-item'),
                playBtn=document.getElementById(uid+'-play'),
                prevBtn=document.getElementById(uid+'-prev'),
                nextBtn=document.getElementById(uid+'-next'),
                bar=document.getElementById(uid+'-bar'),
                barWrap=document.getElementById(uid+'-barwrap'),
                trackEl=document.getElementById(uid+'-track'),
                curEl=document.getElementById(uid+'-cur'),
                durEl=document.getElementById(uid+'-dur'),
                volEl=document.getElementById(uid+'-vol'),
                idx=-1;

            function fmt(s){s=Math.floor(s)||0;return Math.floor(s/60)+':'+(('0'+s%60).slice(-2));}

            function load(i,play){
                if(!items.length)return;
                if(i<0)i=items.length-1;
                if(i>=items.length)i=0;
                idx=i;
                [].forEach.call(items,function(el){el.classList.remove('active');});
                items[idx].classList.add('active');
                items[idx].scrollIntoView({block:'nearest'});
                audio.src=items[idx].getAttribute('data-src');
                trackEl.textContent=items[idx].textContent.trim().replace(/^\d+\.\s*/,'');
                bar.style.width='0%';curEl.textContent='0:00';durEl.textContent='0:00';
                audio.load();
                if(play)audio.play();
            }

            function togglePlay(){
                if(audio.paused){if(idx<0)load(0,true);else audio.play();}
                else audio.pause();
            }

            audio.addEventListener('play',function(){playBtn.innerHTML=svgPause;});
            audio.addEventListener('pause',function(){playBtn.innerHTML=svgPlay;});
            audio.addEventListener('ended',function(){load(idx+1,true);});
            audio.addEventListener('timeupdate',function(){
                if(!audio.duration)return;
                bar.style.width=(audio.currentTime/audio.duration*100)+'%';
                curEl.textContent=fmt(audio.currentTime);
                durEl.textContent=fmt(audio.duration);
            });

            playBtn.addEventListener('click',togglePlay);
            prevBtn.addEventListener('click',function(){load(idx-1,!audio.paused);});
            nextBtn.addEventListener('click',function(){load(idx+1,!audio.paused);});
            barWrap.addEventListener('click',function(e){
                if(!audio.duration)return;
                var r=barWrap.getBoundingClientRect();
                audio.currentTime=((e.clientX-r.left)/r.width)*audio.duration;
            });
            volEl.addEventListener('input',function(){audio.volume=parseFloat(this.value);});
            audio.volume=parseFloat(volEl.value);
            [].forEach.call(items,function(el,i){el.addEventListener('click',function(){load(i,true);});});

            load(0,ap);
        })();
        </script>
        <?php
        return ob_get_clean();
    }

    /**
     * Charge le CSS externe une seule fois par page via une balise <link>.
     */
    private function printCSS()
    {
        static $done = false;
        if ($done) return;
        $done = true;
        // Chemin CSS relatif à la racine web : fonctionne quel que soit DOMAIN
        $cssUrl = '/bl-plugins/mp3player/mp3player.css';
        ?>
        <link rel="stylesheet" href="<?php echo $cssUrl; ?>">
        <?php
    }
}
